unit uParts;

interface

uses
    WinTypes,WinProcs, Classes, Graphics, uDuck, SysUtils,
    uLogoCanvas, uMutate, XMLDoc, XMLIntf;

const
    kRoateCorrectie = 0.25;

type
    ECambrium = class(Exception);
    
    TPart = class;

    TVirtualCreature = class
       public
         function FindPartByID( ID : String ) : TPart; virtual; abstract;
    end;

    TPart = class
      private
        FSize, FR, FPhi : Single;
        FOwner : TVirtualCreature;
        FID : String;
      protected
        function GetAttribute(Tag: String): String; virtual;
        procedure SetAttibute(Tag: String; const Value: String); virtual;

      public
        constructor Create( Owner: TVirtualCreature; ID : String; AR, APhi : Single );
        function CreateClone( NewOwner :TVirtualCreature; Duplicate : Boolean = False )  : TPart; virtual; abstract;
        procedure Draw( Canvas : TLogoCanvas ); virtual; abstract;
        function EnergyCost : Single; virtual;
        procedure AddXmlNodeTo( PartNode : IXMLNode);
        function OutputCount : Integer; virtual;
        function ReadOutput( Idx : Integer ) : Single; virtual;
        function EvalFormula( sFormula : String ) : Single;
        procedure Mutate(FMutationChance : Single); virtual;
        procedure Recalculate; virtual; // evaluate inputs and calculate new outputs

        property R : Single read FR write FR;
        property Size : Single read FSize write FSize;
        property Phi : Single read FPhi write FPhi;
        property ID : String read FID;

        function ListAttributes : String; virtual;// returns attribute names seperated by '|'
        property Attribute[Tag : String] : String read GetAttribute write SetAttibute;
    end;

    TPartList = class(TList)
      private
        FOwner : TVirtualCreature;
        function GetItems(I:Integer) : TPart;
      public
        constructor Create( AnOwner : TVirtualCreature );
        function Add( APart : TPart ) : TPart;
        property Items[ I : Integer ] : TPart read GetItems; default;
        function EnergyCost : Single;
        procedure LoadFromXML( XML : IXMLDocument );
        procedure SaveToXML( XML : IXMLDocument );
        function FindPartByID( ID : String ) : TPart;
    end;

    TEye = class(TPart)
      private
        { Huidige intensiteit }
        FRed, FGreen, FBlue : Single;
      public
        constructor Create( Owner: TVirtualCreature; ID : String; APhi : Single  );
        function CreateClone( NewOwner :TVirtualCreature; Duplicate : Boolean = False ) : TPart; override;
        procedure Draw( Canvas : TLogoCanvas ); override;
        procedure SetColors(R,G,B : Single);
        function OutputCount : Integer; override;
        function ReadOutput( Idx : Integer ) : Single; override;

        property Red : Single read FRed write FRed;
        property Green : Single read FGreen write FGreen;
        property Blue : Single read FBlue write FBlue;
    end;


    TBrain = class(TPart)
      private
        FRandom, FConst  : Single;
        FFormula : String;
        FCurValue : Single;
      protected
        function GetAttribute(Tag: String): String; override;
        procedure SetAttibute(Tag: String; const Value: String); override;
      public
        constructor Create( Owner: TVirtualCreature; ID : String; AR, APhi : Single; ARandom, AConst : Single; AFormula : String );
        function CreateClone( NewOwner :TVirtualCreature; Duplicate : Boolean = False ) : TPart; override;
        procedure Mutate(FMutationChance : Single); override;
        function OutputCount : Integer; override;
        function ReadOutput( Idx : Integer ) : Single; override;
        procedure Draw( Canvas : TLogoCanvas ); override;
        property CurValue : Single read FCurValue;
        function ListAttributes : String; override;
        procedure Recalculate; override;
     end;

    TMouth = class(TPart)
      private
        FEating : Single;
      public
        constructor Create( Owner: TVirtualCreature; ID : String; APhi : Single  );
        function CreateClone( NewOwner :TVirtualCreature; Duplicate : Boolean = False ) : TPart; override;
        procedure Mutate(FMutationChance : Single); override;
        procedure Draw( Canvas : TLogoCanvas ); override;
        procedure Feeding(R : Single);
        property Eating : Single read FEating write FEating;
        function OutputCount : Integer; override;
        function ReadOutput( Idx : Integer ) : Single; override;
    end;

    TEngine = class(TPart)
      private
        FSegCount : Integer;
        FAngle : Single;
        FSpeed : Single;
        FCompr : Single;
        FCount : Single;
        FFormula : String;
      protected
        function GetAttribute(Tag: String): String; override;
        procedure SetAttibute(Tag: String; const Value: String); override;
      public
        constructor Create( Owner: TVirtualCreature; ID : String; APhi, AnAngle : Single; AnFormula : String );
        function EnergyCost: Single; override;
        function CreateClone( NewOwner :TVirtualCreature; Duplicate : Boolean = False ) : TPart; override;
        procedure Mutate(FMutationChance : Single); override;
        procedure GetEffect( var rPush, rRotate : Single; ASize : Single );
        procedure Draw( Canvas : TLogoCanvas ); override;
        function ListAttributes : String; override;
        procedure Recalculate; override;
        property Speed : Single read FSpeed;
    end;

implementation

uses
    uCreature;

{ ___  _   _   _  ___ }
{  |  |_) /_\ |_)  |  }
{  |  |   | | | \  |  }

constructor TPart.Create( Owner: TVirtualCreature; ID : String; AR, APhi : Single );
begin
    inherited Create;
    FOwner := Owner;
    FID := ID;
    FR := AR;
    FPhi := APhi;
    FSize := 0.1;
end;

function TPart.EnergyCost: Single;
begin
    Result := 1;
end;

procedure TPart.AddXmlNodeTo(PartNode: IXMLNode);
var
    AttrName, AttrList : String;
begin
    with PartNode.AddChild( 'part' ) do begin
        AttrList := ListAttributes;
        while AttrList<>'' do begin
            AttrName := ExtractTo(AttrList,'|');
            Attributes[AttrName] := Self.Attribute[AttrName];
        end;
    end;
end;

function TPart.ListAttributes: String;
begin
    Result := 'type|id|r|phi';
end;

function TPart.GetAttribute(Tag: String): String;
begin
    if (Tag = 'type') then
        Result := LowerCase(Copy(ClassName,2,255))
    else if (Tag = 'id') then
        Result := FId
    else if (Tag='r') then
        Result := FloatToStr(FR)
    else if (Tag='phi') then
        Result := FloatToStr(FPhi)
    else
        Result := '';
end;

procedure TPart.SetAttibute(Tag: String; const Value: String);
begin
    if (Tag = 'id') then
        FId := Value
    else if (Tag='r') then
        FR := StrToFloat(Value)
    else if (Tag='phi') then
        FPhi := StrToFloat(Value);
end;

function TPart.OutputCount: Integer;
begin
    Result := 0;
end;

function TPart.ReadOutput(Idx: Integer): Single;
begin
    Result := 0.0;
end;

function TPart.EvalFormula(sFormula: String): Single;
// well, a formula looks very general purpose, but it isn't.
// the structure is:
//  IDOfPart1.InputNr * factor1 + IDOfPart2.InputNr * factor2 ... etc
var
    PartID : String;
    InputNr : Integer;
    Factor : Single;
    Part : TPart;
begin
    Result := 0.0;
    while sFormula <>'' do begin
        PartID := Trim(ExtractTo(sFormula, '.'));
        InputNr := StrToInt( ExtractTo(sFormula, '*' ));
        Factor := StrToFloat(ExtractTo(sFormula, '+' ));
        Part := FOwner.FindPartByID( PartID );
        if Part=nil then
            raise ECambrium.Create( 'TPart.EvalFormula: part not found ('+PartID+')' );
        Result := Result + Factor * Part.ReadOutput(InputNr);
    end;
end;

procedure TPart.Mutate(FMutationChance: Single);
begin
    FPhi := MutateAngle(FPhi,FMutationChance)
end;

procedure TPart.Recalculate;
begin
end;


function TPartList.Add( APart : TPart ) : TPart;
begin
    inherited Add(APart);
    Result := APart;
end;

constructor TPartList.Create(AnOwner: TVirtualCreature);
begin
    inherited Create;
    FOwner := AnOwner;
end;

function TPartList.EnergyCost: Single;
var
    I : Integer;
begin
    Result := 0;
    for I := 0 to Count-1 do
        Result := Result + Items[I].EnergyCost;
end;

function TPartList.FindPartByID(ID: String): TPart;
var
    I : Integer;
begin
    for I := 0 to Count-1 do begin
        if Items[I].ID = ID then begin
            Result := ITems[I];
            Exit;
        end;
    end;
    // nothing found.
    Result := nil;
end;

function TPartList.GetItems(I:Integer) : TPart;
begin
    Result := TObject(inherited Items[I]) as TPart;
end;

procedure TPartList.LoadFromXML(XML: IXMLDocument);
var
    I : Integer;
    PartListNode, PartNode : IXMLNode;
    sType, ID : String;
    Part : TPart;
begin
    PartListNode := XML.DocumentElement.ChildNodes.FindNode('parts');
    for I := 0 to PartListNode.ChildNodes.Count-1 do begin
        PartNode := PartListNode.ChildNodes[I];
        sType := PartNode.Attributes['type'];
        ID := PartNode.Attributes['id'];
        if sType='eye' then
            Part := TEye.Create(FOwner, ID, StrToFloat(PartNode.Attributes['phi']) )
        else if sType = 'engine' then
            Part := TEngine.Create(FOwner, ID, StrToFloat(PartNode.Attributes['phi']),
                                   StrToFloat(PartNode.Attributes['angle']), PartNode.Attributes['formula'])
        else if sType = 'mouth' then
            Part := TMouth.Create(FOwner, ID, StrToFloat(PartNode.Attributes['phi']) )
        else if sType = 'brain' then
            Part := TBrain.Create(FOwner, ID, StrToFloat(PartNode.Attributes['r']),
                                      StrToFloat(PartNode.Attributes['phi']),
                                      StrToFloat(PartNode.Attributes['random']),
                                      StrToFloat(PartNode.Attributes['constant']),
                                      PartNode.Attributes['formula'])
        else
            raise ECambrium.Create( 'TPartList.LoadFromXML: No such part type: ' +sType);

       Add(Part);
    end;
end;

procedure TPartList.SaveToXml( XML : IXMLDocument );
var
    I : Integer;
    PartNode : IXMLNode;
begin
    PartNode := XML.DocumentElement.AddChild('parts');
    for I := 0 to Count-1 do begin
        Items[I].AddXmlNodeTo( PartNode );
    end;
end;


{ ___  _ _ _  _ }
{  |  |_ \/  |_ }
{  |  |_ /   |_ }
constructor TEye.Create( Owner: TVirtualCreature; ID : String; APhi : Single  );
begin
    inherited Create( Owner, ID, 1.0, APhi );
    FSize := FSize *1.8;
end;

function TEye.CreateClone(NewOwner :TVirtualCreature; Duplicate : Boolean) : TPart;
begin
    Result := TEye.Create( NewOwner, ID, Phi );
end;

procedure TEye.Draw( Canvas : TLogoCanvas );
begin
    Canvas.Brush.Color := $02505000;
    Canvas.Circle( FSize );
end;

procedure TEye.SetColors(R,G,B : Single);
begin
    FRed := R;
    FGreen := G;
    FBlue := B;
end;

function TEye.OutputCount: Integer;
begin
    Result := 3;
end;

function TEye.ReadOutput(Idx: Integer): Single;
begin
    case Idx of
        0 : Result := FRed;
        1 : Result := FGreen;
        2 : Result := FBlue;
    end;
end;


{ ___ _  _  _      ___     }
{  |  |\/| | | | |  |  |_| }
{  |  |  | |_| |_|  |  | | }

constructor TMouth.Create( Owner: TVirtualCreature; ID : String; APhi : Single );
begin
    inherited Create( Owner, ID, 1.0, APhi );
    FSize := FSize *2;
end;

function TMouth.CreateClone(NewOwner: TVirtualCreature; Duplicate : Boolean) : TPart;
begin
    Result := TMouth.Create( NewOwner, ID, FPhi );
end;

procedure TMouth.Mutate( FMutationChance : Single );
begin
    FPhi := MutateAngle(FPhi,FMutationChance)
end;

procedure TMouth.Draw( Canvas : TLogoCanvas );
begin
    with Canvas do begin
        Rotate( -Pi/2 );
        Move( FSize/3 );
        Rotate( Pi/2 );
        Draw( FSize );
        Rotate( Pi/2 );
        Draw( FSize*2/3 );
        Rotate( Pi/2 );
        Draw( FSize );
    end;
end;

procedure TMouth.Feeding(R : Single);
begin
    FEating := R;
end;

function TMouth.OutputCount: Integer;
begin
    Result := 1;
end;

function TMouth.ReadOutput(Idx: Integer): Single;
begin
    Result := FEating;
end;

{ ___  _ _  _  _  _ _  _  _ }
{  |  |_ |\ | / _ | |\ | |_ }
{  |  |_ | \| \_/ | | \| |_ }

constructor TEngine.Create( Owner: TVirtualCreature; ID : String; APhi, AnAngle : Single; AnFormula : String );
begin
    inherited Create( Owner, ID, 1.0, APhi );
    FSegCount := 3;
    FCompr := Pi/8;
    FAngle := AnAngle;
    FSpeed := 0;
    FSize := 0.07;
    FFormula := AnFormula;
end;

function TEngine.CreateClone(NewOwner: TVirtualCreature; Duplicate : Boolean) : TPart;
begin
    Result := TEngine.Create( NewOwner, ID, Phi, FAngle, FFormula );
end;

procedure TEngine.Mutate( FMutationChance : Single );
begin
    FPhi := MutateAngle(FPhi,FMutationChance);
    FAngle := MutateAngle(FAngle,FMutationChance);
end;

procedure TEngine.GetEffect( var rPush, rRotate : Single; ASize : Single );
begin
    rPush := Cos(FAngle) *  FSize * FSpeed / ASize;
    rRotate := kRoateCorrectie * Sin(FAngle) * FSize * FSpeed / ASize;
end;

procedure TEngine.Draw( Canvas : TLogoCanvas );
var
    I : Integer;
begin
    Canvas.Rotate( FAngle );
    FCompr := 3*Pi*(1+Sin(FCount/30))/16;
    Canvas.Rotate( FCompr );
    Canvas.Draw( FSize );
    for I := 0 to FSegCount-1 do begin
        Canvas.Rotate( -2*FCompr );
        Canvas.Draw( FSize );
        Canvas.Rotate( 2*FCompr );
        Canvas.Draw( FSize );
    end;
end;

function TEngine.EnergyCost: Single;
begin
    Result := FSpeed/20 +1;
end;

function TEngine.ListAttributes: String;
begin
    Result := (inherited ListAttributes) + '|angle|formula';
end;

function TEngine.GetAttribute(Tag: String): String;
begin
    if (Tag = 'angle') then
        Result := FloatToStr(FAngle)
    else if (Tag='formula') then
        Result := FFormula
    else
        Result := inherited GetAttribute(Tag);
end;

procedure TEngine.SetAttibute(Tag: String; const Value: String);
begin
    if (Tag = 'angle') then
        FAngle := StrToFloat(Value)
    else if (Tag='formula') then
        FFormula := Value
    else
        inherited SetAttibute(Tag,Value);
end;

procedure TEngine.Recalculate;
begin
    if ID='RightEngine' then
        FCount := FCount + 0.001;
    FSpeed := EvalFormula( FFormula );
    if FSpeed<0 then FSpeed := 0.0;
    if FSpeed>100 then FSpeed := 100.0;

    // FCount measures total power on the engine is used to animate the thing.
    FCount := FCount + FSpeed;
end;


{ ___  _   _   _  _ _  _  _   _       }
{  |  |_) |_) /_\ | |\ | | ` |_ |  |  }
{  |  |_) | \ | | | | \| |_, |_ |_ |_ }

constructor TBrain.Create( Owner: TVirtualCreature; ID : String; AR, APhi : Single; ARandom, AConst : Single; AFormula : String );
begin
    inherited Create( Owner, ID, AR, APhi );
    FRandom := ARandom;
    FConst := AConst;
    FFormula := AFormula;
end;

function TBrain.CreateClone(NewOwner : TVirtualCreature; Duplicate : Boolean) : TPart;
begin
    Result := TBrain.Create( NewOwner, ID,
                                 R, Phi, FRandom, FConst,
                                 FFormula );
end;

procedure TBrain.Mutate( FMutationChance : Single );
begin
    FRandom := MutateFactor(FRandom, FMutationChance);
    FConst  := MutateFactor(FConst, FMutationChance);
    FPhi := MutateAngle(FPhi,FMutationChance);
end;


procedure TBrain.Draw( Canvas : TLogoCanvas );
begin
    Canvas.Brush.Color := $303030;
    Canvas.Circle(0.083);
end;

function TBrain.ListAttributes: String;
begin
    Result := (inherited ListAttributes) + '|random|constant|formula';
end;

function TBrain.GetAttribute(Tag: String): String;
begin
    if (Tag = 'random') then
        Result := FloatToStr(FRandom)
    else if (Tag = 'constant') then
        Result := FloatToStr(FConst)
    else if (Tag='formula') then
        Result := FFormula
    else
        Result := inherited GetAttribute(Tag);
end;

procedure TBrain.SetAttibute(Tag: String; const Value: String);
begin
    if (Tag = 'random') then
        FRandom := StrToFloat(Value)
    else if (Tag = 'constant') then
        FConst := StrToFloat(Value)
    else if (Tag='formula') then
        FFormula := Value
    else
        inherited SetAttibute(Tag,Value);
end;


function TBrain.OutputCount: Integer;
begin
    Result := 1;
end;

function TBrain.ReadOutput(Idx: Integer): Single;
begin
    Result := FCurValue;
end;

procedure TBrain.Recalculate;
begin
    FCurValue := Random*FRandom + FConst + EvalFormula( FFormula );
end;



end.
